<?php
require_once '../src/auth.php';
require_once __DIR__ . '/../config/database.php';
protectPage();

/* ===========================================================
   Fetch Exchange Logs with Safe Names and User Names
   =========================================================== */
$query = "
    SELECT 
        e.exchange_id,
        e.amount_from,
        e.exchange_rate,
        e.amount_to,
        e.exchange_date,
        e.notes,
        u.username,
        fs.cashbox_name AS from_safe,
        ts.cashbox_name AS to_safe
    FROM currency_exchanges e
    JOIN users u ON e.user_id = u.user_id
    JOIN cash_boxes fs ON e.from_cashbox_id = fs.cashbox_id
    JOIN cash_boxes ts ON e.to_cashbox_id = ts.cashbox_id
    ORDER BY e.exchange_date DESC
";

$stmt = $pdo->prepare($query);
$stmt->execute();
$logs = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Currency Exchange Log</title>
    <link rel="stylesheet" href="../assets/font-awesome/css/all.min.css">
    <style>
        :root {
            --primary: #2563eb;
            --secondary: #64748b;
            --success: #059669;
            --bg: #f8fafc;
            --border: #e2e8f0;
        }
        body { font-family: 'Inter', sans-serif; background: var(--bg); margin: 0; padding: 30px; color: #1e293b; }
        .container { max-width: 1100px; margin: auto; }
        .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; }
        .header h2 { margin: 0; font-size: 1.5rem; display: flex; align-items: center; gap: 10px; }
        
        .card { background: #fff; border-radius: 12px; box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1); border: 1px solid var(--border); overflow: hidden; }
        table { width: 100%; border-collapse: collapse; text-align: left; }
        th { background: #f1f5f9; padding: 15px; font-size: 12px; text-transform: uppercase; letter-spacing: 0.05em; color: var(--secondary); border-bottom: 2px solid var(--border); }
        td { padding: 15px; font-size: 14px; border-bottom: 1px solid var(--border); }
        tr:hover { background: #f8fafc; }

        .btn { padding: 8px 16px; border-radius: 6px; text-decoration: none; font-size: 13px; font-weight: 600; display: inline-flex; align-items: center; gap: 5px; transition: 0.2s; border: none; cursor: pointer; }
        .btn-back { background: var(--secondary); color: white; }
        .btn-back:hover { background: #475569; }
        
        .badge { padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: bold; }
        .badge-amount { background: #ecfdf5; color: var(--success); }
        .badge-rate { background: #eff6ff; color: var(--primary); }
        
        .text-muted { color: #94a3b8; font-size: 12px; }
        .notes-cell { max-width: 250px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; color: #4b5563; font-style: italic; }
    </style>
</head>
<body>

<div class="container">
    <div class="header">
        <h2><i class="fas fa-history text-primary"></i> Currency Exchange History</h2>
        <a href="safes.php" class="btn btn-back">
            <i class="fas fa-arrow-left"></i> Back to Safes
        </a>
    </div>

    <div class="card">
        <table>
            <thead>
                <tr>
                    <th>Date</th>
                    <th>User</th>
                    <th>From (Source)</th>
                    <th>To (Destination)</th>
                    <th>Exchange Rate</th>
                    <th>Sent Amount</th>
                    <th>Received Amount</th>
                    <th>Notes</th>
                </tr>
            </thead>
            <tbody>
                <?php if (!$logs): ?>
                <tr>
                    <td colspan="8" style="text-align: center; padding: 50px; color: var(--secondary);">
                        <i class="fas fa-folder-open fa-3x" style="display:block; margin-bottom:10px; opacity:0.3;"></i>
                        No exchange records found.
                    </td>
                </tr>
                <?php else: ?>
                    <?php foreach ($logs as $log): ?>
                    <tr>
                        <td>
                            <strong><?= date('M d, Y', strtotime($log['exchange_date'])) ?></strong><br>
                            <span class="text-muted"><?= date('h:i A', strtotime($log['exchange_date'])) ?></span>
                        </td>
                        <td>
                            <i class="fas fa-user-circle"></i> <?= htmlspecialchars($log['username']) ?>
                        </td>
                        <td><span class="badge" style="background:#fee2e2; color:#991b1b;"><?= htmlspecialchars($log['from_safe']) ?></span></td>
                        <td><span class="badge" style="background:#dcfce7; color:#166534;"><?= htmlspecialchars($log['to_safe']) ?></span></td>
                        <td><span class="badge badge-rate">x <?= number_format($log['exchange_rate'], 4) ?></span></td>
                        <td><strong><?= number_format($log['amount_from'], 2) ?></strong></td>
                        <td><span class="badge badge-amount"><?= number_format($log['amount_to'], 2) ?></span></td>
                        <td class="notes-cell" title="<?= htmlspecialchars($log['notes']) ?>">
                            <?= htmlspecialchars($log['notes']) ?: '<span class="text-muted">No notes</span>' ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

</body>
</html>