<?php
session_start();
if (!isset($_SESSION['logged_in'])) {
    header('Location: login.php');
    exit;
}

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/helpers.php';

if (!can('pay', 'view') && !can('pay', 'print')) {
    die('Access Denied');
}

$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) die('Invalid Payment ID');

try {
    $conf = $pdo->query("SELECT * FROM company_settings LIMIT 1")->fetch();

    $stmt = $pdo->prepare("
        SELECT p.*,
               cb.cashbox_name,
               u.full_name AS created_by
        FROM payment_vouchers p
        LEFT JOIN cash_boxes cb ON p.cashbox_id = cb.cashbox_id
        LEFT JOIN users u ON p.user_id = u.user_id
        WHERE p.payment_id = ?
    ");
    $stmt->execute([$id]);
    $v = $stmt->fetch();

    if (!$v) die('Payment voucher not found');
    $is_inactive = (strtolower($v['status']) === 'inactive');

} catch (PDOException $e) {
    die("Database Error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Payment Voucher #<?= htmlspecialchars($v['payment_no'] ?? $v['payment_id']) ?></title>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@400;700&family=Inter:wght@400;700;800&display=swap');

        body {
            background: #f1f5f9;
            font-family: 'Inter', 'Cairo', sans-serif;
            margin: 0;
            padding: 0;
            color: #1e293b;
        }

        .voucher-paper {
            width: 210mm;
            min-height: 297mm;
            background: #fff;
            margin: 20px auto;
            padding: 12mm;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
            position: relative;
            box-sizing: border-box;
            overflow: hidden;
        }

        /* علامة مائية للإلغاء */
        .cancelled-watermark {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%) rotate(-45deg);
            font-size: 150px;
            color: rgba(220, 38, 38, 0.1);
            font-weight: 900;
            border: 20px solid rgba(220, 38, 38, 0.1);
            padding: 20px 60px;
            z-index: 10;
            pointer-events: none;
            text-transform: uppercase;
        }

        /* ===== الهيدر الموحد ===== */
        .main-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            border-bottom: 3px double #1e293b;
            padding-bottom: 15px;
            margin-bottom: 30px;
        }

        .header-content-en { width: 38%; text-align: left; font-size: 13px; line-height: 1.5; }
        .header-content-en h1 { margin: 0 0 5px 0; font-size: 18px; color: #0f172a; font-weight: 800; }

        .header-logo { width: 20%; text-align: center; }
        .header-logo img { max-width: 120px; height: auto; }

        .header-content-ar { width: 38%; text-align: right; direction: rtl; font-size: 14px; line-height: 1.6; }
        .header-content-ar h1 { margin: 0 0 5px 0; font-size: 20px; color: #0f172a; font-weight: 700; }

        /* ===== العنوان والبيانات ===== */
        .title-box {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 40px;
        }
        .title-text {
            font-size: 26px;
            font-weight: 800;
            text-transform: uppercase;
            border-left: 8px solid #be123c; /* لون مائل للأحمر لتمييز الصرف عن القبض */
            padding-left: 15px;
            letter-spacing: 1px;
        }
        .meta-data { text-align: right; font-weight: 600; }

        /* جدول التفاصيل */
        .details-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0 15px;
        }
        .details-table td { padding: 15px; vertical-align: middle; }
        .label-cell {
            width: 160px;
            font-size: 12px;
            font-weight: 700;
            color: #64748b;
            text-transform: uppercase;
        }
        .value-cell {
            background: #f8fafc;
            font-size: 16px;
            font-weight: 600;
            border-bottom: 1px solid #e2e8f0;
        }
        
        /* تمييز مبلغ الصرف */
        .amount-highlight {
            background: #fff1f2 !important; /* لون أحمر هادئ للصرف */
            border: 1px solid #ffe4e6 !important;
        }
        .amount-text {
            font-size: 32px !important;
            font-weight: 800;
            color: #be123c;
            font-family: 'Courier New', monospace;
        }

        /* ===== التوقيعات ===== */
        .signature-section {
            display: flex;
            justify-content: space-around;
            margin-top: 80px;
        }
        .sig-item { width: 250px; text-align: center; }
        .sig-line { border-top: 2px solid #1e293b; margin-bottom: 10px; }
        .sig-label { font-weight: 800; font-size: 13px; text-transform: uppercase; }

        /* الفوتر */
        .footer-info {
            position: absolute;
            bottom: 15mm;
            left: 0;
            right: 0;
            text-align: center;
            font-size: 11px;
            color: #94a3b8;
            border-top: 1px solid #f1f5f9;
            padding-top: 15px;
        }

        .no-print { text-align: center; padding: 20px; background: #0f172a; }
        .btn { padding: 12px 35px; border-radius: 8px; border: none; font-weight: 700; cursor: pointer; margin: 0 5px; }
        .btn-print { background: #10b981; color: white; }
        .btn-back { background: #475569; color: white; }

        @media print {
            body { background: white; }
            .no-print { display: none; }
            .voucher-paper { margin: 0; box-shadow: none; width: 100%; }
        }
    </style>
</head>
<body>

    <div class="no-print">
        <button class="btn btn-print" onclick="window.print()">PRINT VOUCHER</button>
        <button class="btn btn-back" onclick="history.back()">BACK</button>
    </div>

    <div class="voucher-paper">
        <?php if ($is_inactive): ?>
            <div class="cancelled-watermark">VOID</div>
        <?php endif; ?>

        <header class="main-header">
            <div class="header-content-en">
                <h1><?= htmlspecialchars($conf['company_name_en'] ?? 'COMPANY NAME') ?></h1>
                <div>For Customs Clearance & Shipping</div>
                <div><?= htmlspecialchars($conf['address_en'] ?? '') ?></div>
                <div><strong>Tel:</strong> <?= htmlspecialchars($conf['phones_en'] ?? '') ?></div>
            </div>

            <div class="header-logo">
                <?php if(!empty($conf['logo_path'])): ?>
                    <img src="../assets/img/<?= htmlspecialchars($conf['logo_path']) ?>">
                <?php else: ?>
                    <div style="font-weight: bold; border: 1px solid #ccc; padding: 10px;">LOGO</div>
                <?php endif; ?>
            </div>

            <div class="header-content-ar">
                <h1><?= htmlspecialchars($conf['company_name_ar'] ?? 'اسم الشركة') ?></h1>
                <div>للتخليص الجمركي والشحن ونقل البضائع</div>
                <div><?= htmlspecialchars($conf['address_ar'] ?? '') ?></div>
                <div><strong>هاتف:</strong> <?= htmlspecialchars($conf['phones_ar'] ?? '') ?></div>
            </div>
        </header>

        <div class="voucher-body">
            <div class="title-box">
                <div class="title-text" style="<?= $is_inactive ? 'color:#991b1b; border-left-color:#991b1b;' : '' ?>">
                    <?= $is_inactive ? 'CANCELLED PAYMENT' : 'Payment Voucher' ?>
                </div>
                <div class="meta-data">
                    <div style="font-size: 18px;">No: #<?= htmlspecialchars($v['payment_no'] ?? $v['payment_id']) ?></div>
                    <div style="color: #64748b;">Date: <?= date('d M Y', strtotime($v['payment_date'])) ?></div>
                </div>
            </div>

            <table class="details-table">
                <tr>
                    <td class="label-cell">Payment To / Type</td>
                    <td class="value-cell" colspan="3"><?= htmlspecialchars($v['pay_type']) ?></td>
                </tr>
                <tr>
                    <td class="label-cell">Total Amount</td>
                    <td class="value-cell amount-highlight" colspan="3">
                        <span style="font-size: 14px; vertical-align: middle;">USD</span>
                        <span class="amount-text"><?= number_format($v['amount'], 2) ?></span>
                    </td>
                </tr>
                <tr>
                    <td class="label-cell">Reference No.</td>
                    <td class="value-cell" style="width: 30%;"><?= htmlspecialchars($v['ref_id'] ?? '-') ?></td>
                    <td class="label-cell" style="text-align: center; width: 20%;">Paid From</td>
                    <td class="value-cell"><?= htmlspecialchars($v['cashbox_name'] ?? 'Main Cash') ?></td>
                </tr>
                <tr>
                    <td class="label-cell">Description</td>
                    <td class="value-cell" colspan="3" style="min-height: 60px;">
                        <?= nl2br(htmlspecialchars($v['description'] ?? '')) ?>
                    </td>
                </tr>
            </table>

            <?php if ($is_inactive): ?>
                <div style="background: #fee2e2; border: 1px solid #ef4444; padding: 15px; text-align: center; font-weight: bold; color: #991b1b; margin-top: 20px;">
                    THIS PAYMENT VOUCHER IS CANCELLED AND INVALID.
                </div>
            <?php endif; ?>

            <div class="signature-section">
                <div class="sig-item">
                    <div class="sig-line"></div>
                    <div class="sig-label">ACCOUNTS DEPARTMENT<br>قسم الحسابات</div>
                </div>
                <div class="sig-item">
                    <div class="sig-line"></div>
                    <div class="sig-label">GENERAL MANAGER<br>المدير العام</div>
                </div>
            </div>
        </div>

        <div class="footer-info">
            Created By: <?= htmlspecialchars($v['created_by'] ?? '') ?> | 
            <?= htmlspecialchars($conf['address_en'] ?? '') ?> | 
            Tel: <?= htmlspecialchars($conf['phones_en'] ?? '') ?>
        </div>
    </div>
</body>
</html>